<?php

class AppManager {
    private $basePath;
    private $md5Salt;

    public function __construct($basePath, $md5Salt) {
        $this->basePath = rtrim($basePath, '/') . '/';
        $this->md5Salt = $md5Salt;
    }

    private function generateMd5($input) {
        return md5($input . $this->md5Salt);
    }

    // public function getAppId_old($url) {
    //     $cleanUrl = preg_replace('/\?.*/', '', $url);
    //     return $this->generateMd5($cleanUrl);
    // }

    public function getAppId($url) {
        $cleanUrl = preg_replace('/\?.*/', '', $url);
        return $this->generateMd5($cleanUrl);
    }

    public function getFolderId($path) {
        return $this->generateMd5($path);
    }

    public function getFileId($path) {
        return $this->generateMd5($path);
    }

    public function readJson($path) {
        $fullPath = $this->basePath . $path;
        if (file_exists($fullPath)) {
            return json_decode(file_get_contents($fullPath), true);
        }
        return null;
    }

    public function writeJson($path, $data) {
        $fullPath = $this->basePath . $path;
        $dir = dirname($fullPath);
        if (!is_dir($dir)) {
            mkdir($dir, 0777, true);
        }
        file_put_contents($fullPath, json_encode($data, JSON_PRETTY_PRINT));
    }

    public function inflateAccess($access) {
        if (is_array($access)) {
            $access = implode(',', $access);
        }
    
        $accessing = [
            'admin' => 'admin',
            'v' => 'view',
            'view' => 'view',
            'a' => 'add',
            'add' => 'add',
            'm' => 'modify',
            'mod' => 'modify',
            'modify' => 'modify',
            'mv' => 'move',
            'move' => 'move',
            'e' => 'edit',
            'ed' => 'edit',
            'edit' => 'edit',
            'd' => 'dir',
            'dir' => 'dir',
            'c' => 'checklist',
            'check' => 'checklist',
            'ch' => 'checklist',
            'checklist' => 'checklist',
            'de' => 'delete',
            'del' => 'delete',
            'delete' => 'delete',
            'php' => 'php',
        ];
    
        $accessList = [];
        foreach (explode(',', $access) as $a) {
            if (isset($accessing[$a])) {
                $accessList[] = $accessing[$a];
            }
        }
        return $accessList;
    }
    
    

    private function parseUrl($url) {
        $parsedUrl = parse_url($url);
        $query = [];
        if (isset($parsedUrl['query'])) {
            parse_str($parsedUrl['query'], $query);
        }
        return $query;
    }

    public function autoID($identifier) {
        if (strlen($identifier) === 32 && ctype_xdigit($identifier)) {
            return $identifier; // MD5 hash
        } else {
            // Strip all non-numeric characters and format phone number
            $numericPhone = preg_replace('/\D/', '', $identifier);
            return $this->generateMd5($numericPhone);
        }
    }

    public function addFolder_old($identifier, $url, $folderPath = '', $access = '') {
        $query = $this->parseUrl($url);
        if (!$folderPath && isset($query['fo'])) {
            $folderPath = $3query['fo'];
        }
        if (!$folderPath) {
            throw new Exception("Folder path must be specified.");
        }

        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $folderId = $this->getFolderId($folderPath);

        $folderInfo = [
            "path" => $folderPath,
            "access" => $this->inflateAccess($access)
        ];

        $this->writeJson("members/$memberId/apps/$appId/folders/$folderId.json", $folderInfo);
    }

    public function addFile_old($identifier, $url, $filePath = '', $access = '') {
        $query = $this->parseUrl($url);
        if (!$filePath && isset($query['f'])) {
            $filePath = $query['f'];
        }
        if (!$filePath) {
            throw new Exception("File path must be specified.");
        }

        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $fileId = $this->getFileId($filePath);

        $fileInfo = [
            "path" => $filePath,
            "access" => $this->inflateAccess($access)
        ];

        $this->writeJson("members/$memberId/apps/$appId/files/$fileId.json", $fileInfo);
    }

    public function removeFolder($identifier, $url, $folderPath = '') {
        $query = $this->parseUrl($url);
        if (!$folderPath && isset($query['fo'])) {
            $folderPath = $query['fo'];
        }
        if (!$folderPath) {
            throw new Exception("Folder path must be specified.");
        }

        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $folderId = $this->getFolderId($folderPath);

        $fullPath = $this->basePath . "members/$memberId/apps/$appId/folders/$folderId.json";
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }
    }

    public function removeFile($identifier, $url, $filePath = '') {
        $query = $this->parseUrl($url);
        if (!$filePath && isset($query['f'])) {
            $filePath = $query['f'];
        }
        if (!$filePath) {
            throw new Exception("File path must be specified.");
        }

        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $fileId = $this->getFileId($filePath);

        $fullPath = $this->basePath . "members/$memberId/apps/$appId/files/$fileId.json";
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }
    }

    public function addMember($userInfo) {
        $memberId = $this->autoID($userInfo['phone']);
        $this->writeJson("members/$memberId/who.json", $userInfo);
    }

    public function removeMember($identifier) {
        $memberId = $this->autoID($identifier);
        $fullPath = $this->basePath . "members/$memberId";
        if (is_dir($fullPath)) {
            $this->deleteDirectory($fullPath);
        }
    }

    public function add($jsonFile) {
        $data = $this->readJson($jsonFile);

        if (isset($data['userInfo'])) {
            $this->addMember($data['userInfo']);
        }

        if (isset($data['folders'])) {
            foreach ($data['folders'] as $folder) {
                $this->addFolder($data['userInfo']['phone'], $folder['url'], $folder['path'], $folder['access'] ?? '');
            }
        }

        if (isset($data['files'])) {
            foreach ($data['files'] as $file) {
                $this->addFile($data['userInfo']['phone'], $file['url'], $file['path'], $file['access'] ?? '');
            }
        }
    }

    private function deleteDirectory($dir) {
        if (!file_exists($dir)) {
            return true;
        }

        if (!is_dir($dir)) {
            return unlink($dir);
        }

        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }

            if (!$this->deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
                return false;
            }
        }

        return rmdir($dir);
    }

    public function checkAccess($identifier, $url, $filePath = '', $folderPath = '') {
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);

        if ($filePath) {
            $fileId = $this->getFileId($filePath);
            $fileInfo = $this->readJson("members/$memberId/apps/$appId/files/$fileId.json");
            if ($fileInfo && isset($fileInfo['access'])) {
                return $fileInfo['access'];
            }
        }

        if ($folderPath) {
            $folderId = $this->getFolderId($folderPath);
            $folderInfo = $this->readJson("members/$memberId/apps/$appId/folders/$folderId.json");
            if ($folderInfo && isset($folderInfo['access'])) {
                return $folderInfo['access'];
            }
        }

        $appInfo = $this->readJson("members/$memberId/apps/$appId/info.json");
        if ($appInfo && isset($appInfo['access'])) {
            return $appInfo['access'];
        }

        $memberInfo = $this->readJson("members/$memberId/who.json");
        if ($memberInfo && isset($memberInfo['access'])) {
            return $memberInfo['access'];
        }

        return null;
    }

    // New functions
    public function folderList($identifier, $url) {
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);

        $foldersPath = $this->basePath . "members/$memberId/apps/$appId/folders";
        $folders = [];
        if (is_dir($foldersPath)) {
            $folderFiles = scandir($foldersPath);
            foreach ($folderFiles as $file) {
                if ($file === '.' || $file === '..') {
                    continue;
                }
                $folderInfo = $this->readJson("members/$memberId/apps/$appId/folders/$file");
                if ($folderInfo) {
                    $folders[] = $folderInfo;
                }
            }
        }
        return $folders;
    }

    public function fileList($identifier, $url) {
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);

        $filesPath = $this->basePath . "members/$memberId/apps/$appId/files";
        $files = [];
        if (is_dir($filesPath)) {
            $fileFiles = scandir($filesPath);
            foreach ($fileFiles as $file) {
                if ($file === '.' || $file === '..') {
                    continue;
                }
                $fileInfo = $this->readJson("members/$memberId/apps/$appId/files/$file");
                if ($fileInfo) {
                    $files[] = $fileInfo;
                }
            }
        }
        return $files;
    }

    public function appSearch($memberId, $id) {
        $appsPath = $this->basePath . "members/$memberId/apps";
        $apps = scandir($appsPath);

        foreach ($apps as $appId) {
            if ($appId === '.' || $appId === '..') {
                continue;
            }

            $filePath = "$appsPath/$appId/files/$id.json";
            $folderPath = "$appsPath/$appId/folders/$id.json";

            if (file_exists($filePath) || file_exists($folderPath)) {
                return $appId;
            }
        }

        return null;
    }

    public function appList() {
        $appsPath = $this->basePath . "apps";
        $apps = scandir($appsPath);
        $appList = [];

        foreach ($apps as $appId) {
            if ($appId === '.' || $appId === '..') {
                continue;
            }

            $appInfoPath = "$appsPath/$appId/info.json";
            if (file_exists($appInfoPath)) {
                $appInfo = $this->readJson("apps/$appId/info.json");
                if ($appInfo && isset($appInfo['url'])) {
                    $appList[$appId] = $appInfo['url'];
                }
            }
        }

        return $appList;
    }

    public function addAppInfo($url) {
        $appId = $this->getAppId($url);
        $cleanUrl = preg_replace('/\?.*/', '', $url);
        $appInfo = [
            "url" => $cleanUrl
        ];
        $this->writeJson("apps/$appId/info.json", $appInfo);
    }
    
    public function addFolder($identifier, $url, $folderPath = '', $access = '') {
        $query = $this->parseUrl($url);
        if (!$folderPath && isset($query['fo'])) {
            $folderPath = $query['fo'];
        }
        if (!$folderPath) {
            throw new Exception("Folder path must be specified.");
        }
    
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $folderId = $this->getFolderId($folderPath);
    
        $this->addAppInfo($url);
    
        $folderInfo = [
            "path" => $folderPath,
            "access" => $this->inflateAccess($access)
        ];
    
        $this->writeJson("members/$memberId/apps/$appId/folders/$folderId.json", $folderInfo);
    }


    
    
    public function addFile($identifier, $url, $filePath = '', $access = '') {
        $query = $this->parseUrl($url);
        if (!$filePath && isset($query['f'])) {
            $filePath = $query['f'];
        }
        if (!$filePath) {
            throw new Exception("File path must be specified.");
        }
    
        $memberId = $this->autoID($identifier);
        $appId = $this->getAppId($url);
        $fileId = $this->getFileId($filePath);
    
        $this->addAppInfo($url);
    
        $fileInfo = [
            "path" => $filePath,
            "access" => $this->inflateAccess($access)
        ];
    
        $this->writeJson("members/$memberId/apps/$appId/files/$fileId.json", $fileInfo);
    }
    
    public function getBasePath() {
        return $this->basePath;
    }
    
    public function setAppAccess($memberId, $appId, $access) {
        $accessData = $this->inflateAccess(implode(',', $access));
        $appMemberPath = "members/$memberId/apps/$appId/access.json";
        $this->writeJson($appMemberPath, $accessData);
    }

    public function getAppAccess($memberId, $appId) {
        $appMemberPath = "members/$memberId/apps/$appId/access.json";
        $accessData = $this->readJson($appMemberPath);
        return $accessData ? $accessData : [];
    }
    
    public function appGetAdd($memberId, $url, $key, $value) {
        $appId = $this->getAppId($url);
        $keyValueId = md5($key . $value);
        $appGetPath = "members/$memberId/apps/$appId/get/$keyValueId.json";
        $getInfo = [
            "key" => $key,
            "value" => $value
        ];
        $this->writeJson($appGetPath, $getInfo);
    }
    
    

    public function appGetRemove($memberId, $url, $key, $value) {
        $appId = $this->getAppId($url);
        $keyValueId = md5($key . $value);
        $appGetPath = "members/$memberId/apps/$appId/get/$keyValueId.json";
        if (file_exists($appGetPath)) {
            unlink($appGetPath);
        }
    }
    
    
    public function appGetList($memberId, $url) {
        $appId = $this->getAppId($url);
        $getListPath = "members/$memberId/apps/$appId/get";
        $getList = [];
    
        if (is_dir($getListPath)) {
            $getFiles = scandir($getListPath);
            foreach ($getFiles as $getFile) {
                if ($getFile !== '.' && $getFile !== '..') {
                    $getInfo = $this->readJson("$getListPath/$getFile");
                    if ($getInfo) {
                        $getList[] = $getInfo;
                    }
                }
            }
        }
    
        return $getList;
    }
    
    
    
    public function addMemberAppUrl($memberId, $url) {
        $cleanUrl = preg_replace('/\?.*/', '', $url);
        $appId = $this->getAppId($cleanUrl);
        $appInfo = ["url" => $cleanUrl];
        $this->writeJson("apps/$appId/info.json", $appInfo);
        
        // Ensure the member's app directory exists
        $memberAppPath = "members/$memberId/apps/$appId";
        if (!is_dir($memberAppPath)) {
            mkdir($memberAppPath, 0777, true);
        }
    
        // Write info.json for the member's app
        $memberAppInfo = ["url" => $cleanUrl];
        $this->writeJson("$memberAppPath/info.json", $memberAppInfo);
    }
    

    public function setAppGetAccess($memberId, $url, $key, $value, $access) {
        $appId = $this->getAppId($url);
        $keyValueId = md5($key . $value);
        $accessData = $this->inflateAccess($access);
        $appGetAccessPath = "members/$memberId/apps/$appId/get_access/$keyValueId.json";
        $this->writeJson($appGetAccessPath, $accessData);
    }
    
    
    
    
    
    

    public function getAppGetAccess($memberId, $url, $key, $value) {
        $appId = $this->getAppId($url);
        $keyValueId = md5($key . $value);
        $appGetAccessPath = "members/$memberId/apps/$appId/get_access/$keyValueId.json";
        $accessData = $this->readJson($appGetAccessPath);
        return $accessData ? $accessData : [];
    }
    
    
    
    
    public function listFileAccess($memberId, $url, $filePath) {
        $appId = $this->getAppId($url);
        $fileId = $this->getFileId($filePath);
        $fileAccessPath = "members/$memberId/apps/$appId/files_access/$fileId.json";
        return $this->readJson($fileAccessPath);
    }

    public function listFolderAccess($memberId, $url, $folderPath) {
        $appId = $this->getAppId($url);
        $folderId = $this->getFolderId($folderPath);
        $folderAccessPath = "members/$memberId/apps/$appId/folders_access/$folderId.json";
        return $this->readJson($folderAccessPath);
    }

    public function listGetAccess($memberId, $url, $key) {
        $appId = $this->getAppId($url);
        $getAccessPath = "members/$memberId/apps/$appId/get_access/$key.json";
        return $this->readJson($getAccessPath);
    }

    public function listGetAccessWithValue($memberId, $url, $key, $value) {
        $appId = $this->getAppId($url);
        $getAccessPath = "members/$memberId/apps/$appId/get_access/$key.json";
        $accessData = $this->readJson($getAccessPath);
        
        return $accessData ? $accessData : [];
    }
    

}

// // Example usage:
// $basePath = '/path/to/your/data';
// $md5Salt = 'your_salt_string';
// $appManager = new AppManager($basePath, $md5Salt);

// $phone = '813-690-1260';
// $url = 'https://eyeformeta.com/apps/Scrolls/?fo=Family/Starks&f=SELF/travel/co-24-05-shopping.md';
// $filePath = '';
// $folderPath = '';
// $userInfo = [
//     "nickname" => "Scotty",
//     "first" => "Scott",
//     "last" => "Reph",
//     "email" => "scott@softwaredevelopment.solutions",
//     "phone" => '813-690-1260'
// ];

// // Automatically detect ID type
// $detectedId = $appManager->autoID($phone);
// echo "Detected ID: $detectedId\n";

// // Add a member
// $appManager->addMember($userInfo);

// // Add a folder
// $appManager->addFolder($phone, $url, $folderPath, 'admin,view,add');

// // Add a file
// $appManager->addFile($phone, $url, $filePath, 'view,edit');

// // List all folders
// $folders = $appManager->folderList($phone, $url);
// print_r($folders);

// // List all files
// $files = $appManager->fileList($phone, $url);
// print_r($files);
