-- Disable foreign key checks to drop tables
SET FOREIGN_KEY_CHECKS = 0;

-- Drop all existing tables
DROP TABLE IF EXISTS CharacterBattles;
DROP TABLE IF EXISTS CharacterItems;
DROP TABLE IF EXISTS CharacterSpells;
DROP TABLE IF EXISTS NPCInteractions;
DROP TABLE IF EXISTS Narratives;
DROP TABLE IF EXISTS Storylines;
DROP TABLE IF EXISTS Battles;
DROP TABLE IF EXISTS Items;
DROP TABLE IF EXISTS Spells;
DROP TABLE IF EXISTS Characters;
DROP TABLE IF EXISTS Campaigns;
DROP TABLE IF EXISTS DungeonMasters;
DROP TABLE IF EXISTS Players;

-- Re-enable foreign key checks
SET FOREIGN_KEY_CHECKS = 1;

-- Create Players Table
CREATE TABLE Players (
    PlayerID INT AUTO_INCREMENT PRIMARY KEY,
    Name VARCHAR(255) NOT NULL,
    Email VARCHAR(255) NOT NULL UNIQUE,
    Password VARCHAR(255) NOT NULL,
    CreatedAt TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Create DungeonMasters Table
CREATE TABLE DungeonMasters (
    DungeonMasterID INT AUTO_INCREMENT PRIMARY KEY,
    Name VARCHAR(255) NOT NULL,
    Email VARCHAR(255) NOT NULL UNIQUE,
    ExperienceLevel VARCHAR(50),
    CreatedAt TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Create Campaigns Table
CREATE TABLE Campaigns (
    CampaignID INT AUTO_INCREMENT PRIMARY KEY,
    Name VARCHAR(255) NOT NULL,
    Description TEXT,
    DungeonMasterID INT NOT NULL,
    StartDate DATE NOT NULL,
    EndDate DATE,
    Status VARCHAR(50) NOT NULL,
    FOREIGN KEY (DungeonMasterID) REFERENCES DungeonMasters(DungeonMasterID)
);

-- Create Characters Table
CREATE TABLE Characters (
    CharacterID INT AUTO_INCREMENT PRIMARY KEY,
    Name VARCHAR(255) NOT NULL,
    Class VARCHAR(50) NOT NULL,
    Level INT NOT NULL CHECK (Level >= 1),
    Race VARCHAR(50) NOT NULL,
    Background VARCHAR(255),
    PlayerID INT NOT NULL,
    Player VARCHAR(255),
    CampaignID INT,
    Health INT NOT NULL CHECK (Health >= 0),
    Experience INT DEFAULT 0,
    Alignment VARCHAR(50),
    FOREIGN KEY (PlayerID) REFERENCES Players(PlayerID),
    FOREIGN KEY (CampaignID) REFERENCES Campaigns(CampaignID)
);

-- Create Spells Table
CREATE TABLE Spells (
    SpellID INT AUTO_INCREMENT PRIMARY KEY,
    Name VARCHAR(255) NOT NULL,
    Description TEXT,
    Level INT NOT NULL CHECK (Level >= 0),
    CastingTime VARCHAR(50),
    SpellRange VARCHAR(50),
    Components VARCHAR(255),
    Duration VARCHAR(50)
);

-- Create CharacterSpells Junction Table
CREATE TABLE CharacterSpells (
    CharacterSpellID INT AUTO_INCREMENT PRIMARY KEY,
    CharacterID INT NOT NULL,
    SpellID INT NOT NULL,
    FOREIGN KEY (CharacterID) REFERENCES Characters(CharacterID),
    FOREIGN KEY (SpellID) REFERENCES Spells(SpellID)
);

-- Create Items Table
CREATE TABLE Items (
    ItemID INT AUTO_INCREMENT PRIMARY KEY,
    Name VARCHAR(255) NOT NULL,
    Type VARCHAR(50),
    Description TEXT,
    Weight DECIMAL(10, 2),
    Value DECIMAL(10, 2)
);

-- Create CharacterItems Junction Table
CREATE TABLE CharacterItems (
    CharacterItemID INT AUTO_INCREMENT PRIMARY KEY,
    CharacterID INT NOT NULL,
    ItemID INT NOT NULL,
    Quantity INT NOT NULL CHECK (Quantity >= 0),
    FOREIGN KEY (CharacterID) REFERENCES Characters(CharacterID),
    FOREIGN KEY (ItemID) REFERENCES Items(ItemID)
);

-- Create Battles Table
CREATE TABLE Battles (
    BattleID INT AUTO_INCREMENT PRIMARY KEY,
    CampaignID INT NOT NULL,
    Name VARCHAR(255) NOT NULL,
    Description TEXT,
    Outcome VARCHAR(255),
    Date DATE NOT NULL,
    FOREIGN KEY (CampaignID) REFERENCES Campaigns(CampaignID)
);

-- Create CharacterBattles Junction Table
CREATE TABLE CharacterBattles (
    CharacterBattleID INT AUTO_INCREMENT PRIMARY KEY,
    CharacterID INT NOT NULL,
    BattleID INT NOT NULL,
    Role VARCHAR(255),
    Outcome VARCHAR(255),
    FOREIGN KEY (CharacterID) REFERENCES Characters(CharacterID),
    FOREIGN KEY (BattleID) REFERENCES Battles(BattleID)
);

-- Create Narratives Table
CREATE TABLE Narratives (
    NarrativeID INT AUTO_INCREMENT PRIMARY KEY,
    CampaignID INT NOT NULL,
    Title VARCHAR(255) NOT NULL,
    Description TEXT,
    Date DATE NOT NULL,
    FOREIGN KEY (CampaignID) REFERENCES Campaigns(CampaignID)
);

-- Create NPCInteractions Table
CREATE TABLE NPCInteractions (
    InteractionID INT AUTO_INCREMENT PRIMARY KEY,
    NPCID INT NOT NULL,
    PlayerCharacterID INT NOT NULL,
    Description TEXT,
    Date DATE,
    Outcome VARCHAR(255),
    FOREIGN KEY (NPCID) REFERENCES Characters(CharacterID),
    FOREIGN KEY (PlayerCharacterID) REFERENCES Characters(CharacterID)
);

-- Create Storylines Table
CREATE TABLE Storylines (
    StorylineID INT AUTO_INCREMENT PRIMARY KEY,
    CampaignID INT NOT NULL,
    Title VARCHAR(255) NOT NULL,
    StartDate DATE NOT NULL,
    EndDate DATE,
    Status VARCHAR(50) NOT NULL,
    FOREIGN KEY (CampaignID) REFERENCES Campaigns(CampaignID)
);
