-- Core table for character classes
CREATE TABLE character_classes (
    class_id INT AUTO_INCREMENT PRIMARY KEY,
    class_name VARCHAR(255) NOT NULL UNIQUE,
    hit_dice VARCHAR(10) NOT NULL,
    primary_abilities JSON NOT NULL, -- Example: ["Strength", "Constitution"]
    saving_throw_proficiencies JSON NOT NULL, -- Example: ["Strength", "Dexterity"]
    skill_proficiencies JSON NOT NULL, -- Example: ["Athletics", "Perception"]
    spellcasting_ability VARCHAR(50), -- Example: "Wisdom"
    class_features JSON -- Example: {"Level 1": ["Rage", "Unarmored Defense"]}
);

-- Table for character races
CREATE TABLE character_races (
    race_id INT AUTO_INCREMENT PRIMARY KEY,
    race_name VARCHAR(255) NOT NULL UNIQUE,
    ability_score_increase JSON NOT NULL, -- Example: {"Strength": 2, "Constitution": 1}
    speed INT NOT NULL, -- Example: 30 feet
    traits JSON -- Example: {"Darkvision": "60 feet", "Savage Attacks": "Extra damage on crits"}
);

-- Table for character features (abilities and traits unique to classes and races)
CREATE TABLE character_features (
    feature_id INT AUTO_INCREMENT PRIMARY KEY,
    feature_name VARCHAR(255) NOT NULL,
    description TEXT NOT NULL,
    level_required INT DEFAULT 1,
    class_id INT,
    race_id INT,
    FOREIGN KEY (class_id) REFERENCES character_classes(class_id),
    FOREIGN KEY (race_id) REFERENCES character_races(race_id)
);

-- Table for character skills
CREATE TABLE character_skills (
    skill_id INT AUTO_INCREMENT PRIMARY KEY,
    skill_name VARCHAR(255) NOT NULL UNIQUE,
    associated_ability VARCHAR(50) NOT NULL -- Example: "Dexterity"
);

-- Table for character proficiencies (weapons, armor, tools, etc.)
CREATE TABLE character_proficiencies (
    proficiency_id INT AUTO_INCREMENT PRIMARY KEY,
    proficiency_name VARCHAR(255) NOT NULL UNIQUE,
    type VARCHAR(50) NOT NULL -- Example: "Weapon", "Armor", "Tool"
);

-- Table for character equipment
CREATE TABLE character_equipment (
    equipment_id INT AUTO_INCREMENT PRIMARY KEY,
    equipment_name VARCHAR(255) NOT NULL UNIQUE,
    type VARCHAR(50) NOT NULL, -- Example: "Weapon", "Armor", "Gear"
    properties JSON -- Example: {"Weight": "10 lb", "Cost": "15 gp"}
);

-- Table for character backgrounds
CREATE TABLE character_backgrounds (
    background_id INT AUTO_INCREMENT PRIMARY KEY,
    background_name VARCHAR(255) NOT NULL UNIQUE,
    skill_proficiencies JSON NOT NULL, -- Example: ["Insight", "Religion"]
    equipment JSON NOT NULL, -- Example: {"Items": ["Holy Symbol", "Prayer Book"]}
    feature VARCHAR(255), -- Example: "Shelter of the Faithful"
    description TEXT -- Details about the background
);

-- Table for character spells
CREATE TABLE character_spells (
    spell_id INT AUTO_INCREMENT PRIMARY KEY,
    spell_name VARCHAR(255) NOT NULL UNIQUE,
    level INT NOT NULL, -- Spell level
    school VARCHAR(50) NOT NULL, -- Example: "Evocation"
    casting_time VARCHAR(50) NOT NULL, -- Example: "1 Action"
    _range VARCHAR(50) NOT NULL, -- Example: "30 feet"
    components JSON NOT NULL, -- Example: {"Verbal": true, "Somatic": true, "Material": "Diamond worth 50 gp"}
    duration VARCHAR(50) NOT NULL, -- Example: "Concentration, up to 1 minute"
    description TEXT NOT NULL -- Spell effect details
);


-- Insert data for character classes
INSERT INTO character_classes (class_name, hit_dice, primary_abilities, saving_throw_proficiencies, skill_proficiencies, spellcasting_ability, class_features)
VALUES
    ("Barbarian", "d12", '["Strength", "Constitution"]', '["Strength", "Constitution"]', '["Athletics", "Survival"]', NULL, '{"Level 1": ["Rage", "Unarmored Defense"]}'),
    ("Bard", "d8", '["Charisma"]', '["Dexterity", "Charisma"]', '["Performance", "Persuasion"]', "Charisma", '{"Level 1": ["Bardic Inspiration"]}'),
    ("Cleric", "d8", '["Wisdom"]', '["Wisdom", "Charisma"]', '["Medicine", "Insight"]', "Wisdom", '{"Level 1": ["Spellcasting", "Divine Domain"]}'),
    ("Druid", "d8", '["Wisdom"]', '["Intelligence", "Wisdom"]', '["Nature", "Animal Handling"]', "Wisdom", '{"Level 1": ["Spellcasting", "Wild Shape"]}'),
    ("Fighter", "d10", '["Strength", "Constitution"]', '["Strength", "Constitution"]', '["Athletics", "Intimidation"]', NULL, '{"Level 1": ["Second Wind"]}'),
    ("Monk", "d8", '["Dexterity", "Wisdom"]', '["Strength", "Dexterity"]', '["Acrobatics", "Stealth"]', NULL, '{"Level 1": ["Unarmored Defense", "Martial Arts"]}'),
    ("Paladin", "d10", '["Strength", "Charisma"]', '["Wisdom", "Charisma"]', '["Religion", "Intimidation"]', "Charisma", '{"Level 1": ["Divine Sense", "Lay on Hands"]}'),
    ("Ranger", "d10", '["Dexterity", "Wisdom"]', '["Strength", "Dexterity"]', '["Survival", "Perception"]', NULL, '{"Level 1": ["Favored Enemy", "Natural Explorer"]}'),
    ("Rogue", "d8", '["Dexterity"]', '["Dexterity", "Intelligence"]', '["Stealth", "Acrobatics"]', NULL, '{"Level 1": ["Sneak Attack", "Thieves\' Cant"]}'),
    ("Sorcerer", "d6", '["Charisma"]', '["Constitution", "Charisma"]', '["Arcana", "Deception"]', "Charisma", '{"Level 1": ["Spellcasting", "Sorcerous Origin"]}'),
    ("Warlock", "d8", '["Charisma"]', '["Wisdom", "Charisma"]', '["Intimidation", "Arcana"]', "Charisma", '{"Level 1": ["Pact Magic", "Eldritch Invocations"]}'),
    ("Wizard", "d6", '["Intelligence"]', '["Intelligence", "Wisdom"]', '["Arcana", "History"]', "Intelligence", '{"Level 1": ["Spellcasting", "Arcane Recovery"]}');

-- Insert data for character races
INSERT INTO character_races (race_name, ability_score_increase, speed, traits)
VALUES
    ("Human", '{"All": 1}', 30, '{"Versatile": "Bonus skill"}'),
    ("Elf", '{"Dexterity": 2}', 30, '{"Darkvision": "60 feet", "Keen Senses": "Proficient in Perception"}'),
    ("Dwarf", '{"Constitution": 2}', 25, '{"Darkvision": "60 feet", "Dwarven Resilience": "Advantage on poison saves"}'),
    ("Halfling", '{"Dexterity": 2}', 25, '{"Lucky": "Reroll 1s", "Brave": "Advantage on fear saves"}');

-- Insert data for character skills
INSERT INTO character_skills (skill_name, associated_ability)
VALUES
    ("Acrobatics", "Dexterity"),
    ("Animal Handling", "Wisdom"),
    ("Arcana", "Intelligence"),
    ("Athletics", "Strength"),
    ("Deception", "Charisma"),
    ("History", "Intelligence"),
    ("Insight", "Wisdom"),
    ("Intimidation", "Charisma"),
    ("Investigation", "Intelligence"),
    ("Medicine", "Wisdom"),
    ("Nature", "Intelligence"),
    ("Perception", "Wisdom"),
    ("Performance", "Charisma"),
    ("Persuasion", "Charisma"),
    ("Religion", "Intelligence"),
    ("Sleight of Hand", "Dexterity"),
    ("Stealth", "Dexterity"),
    ("Survival", "Wisdom");

-- Insert data for character proficiencies
INSERT INTO character_proficiencies (proficiency_name, type)
VALUES
    ("Light Armor", "Armor"),
    ("Medium Armor", "Armor"),
    ("Heavy Armor", "Armor"),
    ("Simple Weapons", "Weapon"),
    ("Martial Weapons", "Weapon"),
    ("Thieves' Tools", "Tool"),
    ("Herbalism Kit", "Tool"),
    ("Musical Instrument", "Tool");

-- Insert data for character equipment
INSERT INTO character_equipment (equipment_name, type, properties)
VALUES
    ("Longsword", "Weapon", '{"Weight": "3 lb", "Cost": "15 gp", "Damage": "1d8 slashing", "Properties": ["Versatile"]}'),
    ("Shortsword", "Weapon", '{"Weight": "2 lb", "Cost": "10 gp", "Damage": "1d6 piercing", "Properties": ["Finesse", "Light"]}'),
    ("Greatsword", "Weapon", '{"Weight": "6 lb", "Cost": "50 gp", "Damage": "2d6 slashing", "Properties": []}'),
    ("Chainmail", "Armor", '{"Weight": "55 lb", "Cost": "75 gp", "AC": "16", "Properties": ["Disadvantage on Stealth"]}'),
    ("Leather Armor", "Armor", '{"Weight": "10 lb", "Cost": "10 gp", "AC": "11 + Dex modifier", "Properties": []}'),
    ("Explorer's Pack", "Gear", '{"Weight": "59 lb", "Cost": "10 gp", "Contents": ["Backpack", "Bedroll", "Rations (10 days)", "Rope (50 feet)"]}'),
    ("Dagger", "Weapon", '{"Weight": "1 lb", "Cost": "2 gp", "Damage": "1d4 piercing", "Properties": ["Finesse", "Light", "Thrown (range 20/60)"]}'),
    ("Shield", "Armor", '{"Weight": "6 lb", "Cost": "10 gp", "AC": "+2", "Properties": []}');

-- Insert data for character backgrounds
INSERT INTO character_backgrounds (background_name, skill_proficiencies, equipment, feature, description)
VALUES
    ("Acolyte", '["Insight", "Religion"]', '{"Items": ["Holy Symbol", "Prayer Book", "5 sticks of incense", "Vestments", "Common Clothes"]}', "Shelter of the Faithful", "You command the respect of those who share your faith."),
    ("Criminal", '["Deception", "Stealth"]', '{"Items": ["Crowbar", "Dark Clothes"]}', "Criminal Contact", "You have a reliable and trustworthy contact who acts as your liaison to a network of criminals."),
    ("Folk Hero", '["Animal Handling", "Survival"]', '{"Items": ["Shovel", "Iron Pot"]}', "Rustic Hospitality", "You can find a place to hide, rest, or recuperate among common folk.");

-- Insert data for character spells
INSERT INTO character_spells (spell_name, level, school, casting_time, _range, components, duration, description)
VALUES
    ("Fireball", 3, "Evocation", "1 Action", "150 feet", '{"Verbal": true, "Somatic": true, "Material": "A tiny ball of bat guano and sulfur"}', "Instantaneous", "A bright streak flashes from your pointing finger to a point you choose."),
    ("Mage Hand", 0, "Conjuration", "1 Action", "30 feet", '{"Verbal": true, "Somatic": true}', "1 minute", "A spectral, floating hand appears at a point you choose within range."),
    ("Healing Word", 1, "Evocation", "1 Bonus Action", "60 feet", '{"Verbal": true}', "Instantaneous", "A creature of your choice you can see within range regains hit points.");

